<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * @property int $id
 * @property int $user_id
 * @property int $schedule_id
 * @property string $game_category
 * @property string $game_name
 * @property string $date
 * @property string $start_time
 * @property string $address
 * @property float $price
 * @property string $full_name
 * @property string $email
 * @property string $phone
 * @property string $payment_method
 * @property string $player_level
 * @property int $personal_data_agreement
 * @property string $status
 * @property string $created_at
 * @property string $updated_at
 *
 * @property User $user
 * @property Schedule $schedule
 */
class Application extends ActiveRecord
{
    const STATUS_NEW = 'new';
    const STATUS_APPROVED = 'approved';
    const STATUS_COMPLETED = 'completed';

    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';
    const PAYMENT_TRANSFER = 'transfer';

    const LEVEL_BEGINNER = 'beginner';
    const LEVEL_AMATEUR = 'amateur';
    const LEVEL_PROFESSIONAL = 'professional';

    // Добавляем свойство для хранения выбранного расписания
    public $selected_schedule_id;

    public static function tableName()
    {
        return '{{%applications}}';
    }

    public function rules()
    {
        return [
            [['schedule_id', 'game_name', 'payment_method', 'player_level', 'personal_data_agreement'], 'required'],
            [['user_id', 'schedule_id'], 'integer'],
            [['price'], 'number'],
            [['date', 'start_time'], 'safe'],
            [['personal_data_agreement'], 'boolean'],
            [['game_category', 'game_name', 'address', 'full_name', 'email', 'phone'], 'string', 'max' => 255],
            [['payment_method', 'player_level', 'status'], 'string', 'max' => 20],
            ['personal_data_agreement', 'compare', 'compareValue' => true, 'message' => 'Вы должны согласиться на обработку персональных данных'],
            ['user_id', 'exist', 'skipOnError' => true, 'targetClass' => User::class, 'targetAttribute' => ['user_id' => 'id']],
            ['schedule_id', 'exist', 'skipOnError' => true, 'targetClass' => Schedule::class, 'targetAttribute' => ['schedule_id' => 'id']],
            ['status', 'default', 'value' => self::STATUS_NEW],
            ['status', 'in', 'range' => array_keys(self::getStatuses())],
        ];
    }

    public function attributeLabels()
    {
        return [
            'schedule_id' => 'Расписание',
            'game_category' => 'Категория игры',
            'game_name' => 'Название игры',
            'date' => 'Дата',
            'start_time' => 'Время начала',
            'address' => 'Адрес',
            'price' => 'Стоимость',
            'full_name' => 'ФИО',
            'email' => 'E-mail',
            'phone' => 'Телефон',
            'payment_method' => 'Способ оплаты',
            'player_level' => 'Уровень игрока',
            'personal_data_agreement' => 'Согласие на обработку персональных данных',
        ];
    }

    public static function getStatuses()
    {
        return [
            self::STATUS_NEW => 'Новая',
            self::STATUS_APPROVED => 'Заявка одобрена',
            self::STATUS_COMPLETED => 'Заказ завершен',
        ];
    }

    public static function getPaymentMethods()
    {
        return [
            self::PAYMENT_CASH => 'Наличными',
            self::PAYMENT_CARD => 'Картой',
            self::PAYMENT_TRANSFER => 'Переводом',
        ];
    }

    public static function getPlayerLevels()
    {
        return [
            self::LEVEL_BEGINNER => 'Новичок',
            self::LEVEL_AMATEUR => 'Любитель',
            self::LEVEL_PROFESSIONAL => 'Профессионал',
        ];
    }

    public function getStatusLabel()
    {
        $statuses = self::getStatuses();
        return $statuses[$this->status] ?? $this->status;
    }

    public function getPaymentMethodLabel()
    {
        $methods = self::getPaymentMethods();
        return $methods[$this->payment_method] ?? $this->payment_method;
    }

    public function getPlayerLevelLabel()
    {
        $levels = self::getPlayerLevels();
        return $levels[$this->player_level] ?? $this->player_level;
    }

    public function getUser()
    {
        return $this->hasOne(User::class, ['id' => 'user_id']);
    }

    public function getSchedule()
    {
        return $this->hasOne(Schedule::class, ['id' => 'schedule_id']);
    }

    public function beforeSave($insert)
    {
        if ($insert) {
            $this->user_id = Yii::$app->user->id;
        }
        return parent::beforeSave($insert);
    }
}