<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\Response;
use yii\filters\AccessControl;
use app\models\Game;
use app\models\Schedule;
use app\models\Review;
use app\models\Subscription;
use app\models\Application;
use app\models\User;
use yii\data\ActiveDataProvider;
use yii\web\NotFoundHttpException;

class SiteController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'only' => ['profile', 'applications', 'create-application', 'view-application', 'cancel-application', 'add-review'],
                'rules' => [
                    [
                        'actions' => ['profile', 'applications', 'create-application', 'view-application', 'cancel-application', 'add-review'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
        ];
    }

    public function actionIndex()
    {
        $popularGames = Game::find()->where(['is_popular' => true])->all();
        $upcomingSchedule = Schedule::find()
            ->where(['>=', 'date', date('Y-m-d')])
            ->andWhere(['<=', 'date', date('Y-m-d', strtotime('+30 days'))])
            ->orderBy(['date' => SORT_ASC, 'start_time' => SORT_ASC])
            ->all();
        $reviews = Review::find()->with('user')->orderBy(['created_at' => SORT_DESC])->limit(5)->all();

        // Создаем модель для подписки
        $subscriptionModel = new Subscription();

        return $this->render('index', [
            'popularGames' => $popularGames,
            'upcomingSchedule' => $upcomingSchedule,
            'reviews' => $reviews,
            'subscriptionModel' => $subscriptionModel,
        ]);
    }

    public function actionAbout()
    {
        return $this->render('about');
    }

    public function actionCatalog()
    {
        $games = Game::find()->all();
        $gamesByCategory = [];
        foreach ($games as $game) {
            $gamesByCategory[$game->category][] = $game;
        }

        return $this->render('catalog', [
            'gamesByCategory' => $gamesByCategory,
        ]);
    }

    public function actionRegister()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new User();
        $model->scenario = 'register';

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Регистрация успешно завершена!');

            // Автоматическая авторизация после регистрации
            Yii::$app->user->login($model, 3600*24*30);

            return $this->goHome();
        }

        return $this->render('register', [
            'model' => $model,
        ]);
    }

    public function actionLogin()
    {
        if (!Yii::$app->user->isGuest) {
            return $this->goHome();
        }

        $model = new User();
        $model->scenario = 'login';

        if ($model->load(Yii::$app->request->post())) {
            // Ищем пользователя по телефону
            $user = User::findByPhone($model->phone);

            if ($user && $user->validatePassword($model->password)) {
                // Авторизуем пользователя
                Yii::$app->user->login($user, $model->rememberMe ? 3600*24*30 : 0);
                Yii::$app->session->setFlash('success', 'Вы успешно авторизовались!');
                return $this->goHome();
            } else {
                $model->addError('password', 'Неверный телефон или пароль');
            }
        }

        return $this->render('login', [
            'model' => $model,
        ]);
    }

    public function actionLogout()
    {
        Yii::$app->user->logout();
        return $this->goHome();
    }

    public function actionProfile()
    {
        $user = Yii::$app->user->identity;
        $applications = Application::find()
            ->where(['user_id' => $user->id])
            ->orderBy(['created_at' => SORT_DESC])
            ->all();

        return $this->render('profile', [
            'user' => $user,
            'applications' => $applications,
        ]);
    }

    public function actionApplications()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Application::find()->where(['user_id' => Yii::$app->user->id]),
            'sort' => ['defaultOrder' => ['created_at' => SORT_DESC]],
        ]);

        return $this->render('applications', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionCreateApplication($schedule_id = null)
    {
        if (Yii::$app->user->isGuest) {
            return $this->redirect(['site/login']);
        }

        $model = new Application();
        $user = Yii::$app->user->identity;

        // Если передали schedule_id, заполняем форму
        if ($schedule_id) {
            $schedule = Schedule::findOne($schedule_id);
            if ($schedule && !$schedule->isFull()) {
                $model->schedule_id = $schedule->id;
                $model->game_category = $schedule->game->category;
                $model->game_name = $schedule->game->name;
                $model->date = $schedule->date;
                $model->start_time = $schedule->start_time;
                $model->address = $schedule->address;
                $model->price = $schedule->price;
            } else {
                Yii::$app->session->setFlash('error', 'Выбранная игра недоступна или все места заняты.');
                return $this->redirect(['site/create-application']);
            }
        }

        // Заполняем данные пользователя
        if ($user) {
            $model->full_name = $user->full_name;
            $model->email = $user->email;
            $model->phone = $user->phone;
        }

        // Обработка POST-запроса
        if ($model->load(Yii::$app->request->post())) {
            // Валидируем форму
            if ($model->validate()) {
                // Проверяем, что расписание еще доступно
                $schedule = Schedule::findOne($model->schedule_id);
                if (!$schedule || $schedule->isFull()) {
                    Yii::$app->session->setFlash('error', 'Выбранная игра недоступна или все места заняты.');
                    return $this->refresh();
                }

                // Сохраняем заявку
                if ($model->save()) {
                    // Увеличиваем счетчик участников
                    $schedule->current_players += 1;
                    $schedule->save();

                    Yii::$app->session->setFlash('success', 'Заявка успешно создана!');
                    return $this->redirect(['site/applications']);
                }
            }
        }

        // Получаем доступные расписания
        $schedules = Schedule::find()
            ->where(['>=', 'date', date('Y-m-d')])
            ->andWhere(['<=', 'date', date('Y-m-d', strtotime('+30 days'))])
            ->with('game')
            ->orderBy(['date' => SORT_ASC, 'start_time' => SORT_ASC])
            ->all();

        // Фильтруем только доступные игры (где есть места)
        $availableSchedules = [];
        foreach ($schedules as $schedule) {
            if (!$schedule->isFull()) {
                $availableSchedules[] = $schedule;
            }
        }

        return $this->render('create-application', [
            'model' => $model,
            'schedules' => $availableSchedules,
            'user' => $user,
        ]);
    }

    public function actionViewApplication($id)
    {
        $model = Application::findOne($id);

        if (!$model || $model->user_id !== Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        return $this->render('view-application', [
            'model' => $model,
        ]);
    }

    public function actionCancelApplication($id)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $model = Application::findOne($id);

        if ($model && $model->user_id == Yii::$app->user->id && $model->status === Application::STATUS_NEW) {
            // Уменьшаем счетчик участников
            $schedule = $model->schedule;
            if ($schedule) {
                $schedule->current_players -= 1;
                $schedule->save();
            }

            if ($model->delete()) {
                return ['success' => true];
            }
        }

        return ['success' => false, 'message' => 'Не удалось отменить заявку'];
    }

    public function actionAddReview()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $review = new Review();
        $review->user_id = Yii::$app->user->id;
        $review->application_id = Yii::$app->request->post('application_id');
        $review->rating = Yii::$app->request->post('rating');
        $review->comment = Yii::$app->request->post('comment');

        if ($review->save()) {
            return ['success' => true];
        }

        return ['success' => false, 'errors' => $review->errors];
    }

    public function actionSubscribe()
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        $model = new Subscription();
        if ($model->load(Yii::$app->request->post(), '') && $model->save()) {
            return ['success' => true, 'message' => 'Вы успешно подписались на рассылку!'];
        }

        return ['success' => false, 'errors' => $model->errors];
    }

    public function actionContact()
    {
        return $this->render('contact');
    }
}