<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Application;
use app\models\User;
use app\models\Schedule;
use app\models\Game;
use app\models\Review;
use yii\web\NotFoundHttpException;

class AdminController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function ($rule, $action) {
                            $user = Yii::$app->user->identity;
                            return $user && $user->phone === '89999999999';
                        }
                    ],
                ],
                'denyCallback' => function ($rule, $action) {
                    return $this->redirect(['site/login']);
                },
            ],
        ];
    }

    public function actionIndex()
    {
        $totalApplications = Application::find()->count();
        $newApplications = Application::find()->where(['status' => Application::STATUS_NEW])->count();
        $totalUsers = User::find()->count();
        $upcomingGames = Schedule::find()
            ->where(['>=', 'date', date('Y-m-d')])
            ->count();

        return $this->render('index', [
            'totalApplications' => $totalApplications,
            'newApplications' => $newApplications,
            'totalUsers' => $totalUsers,
            'upcomingGames' => $upcomingGames,
        ]);
    }

    public function actionApplications()
    {
        $query = Application::find()->joinWith(['user', 'schedule']);

        // Фильтрация по статусу
        $status = Yii::$app->request->get('status');
        if ($status) {
            $query->andWhere(['status' => $status]);
        }

        // Фильтрация по дате
        $date = Yii::$app->request->get('date');
        if ($date) {
            $query->andWhere(['date' => $date]);
        }

        // Поиск
        $search = Yii::$app->request->get('search');
        if ($search) {
            $query->andWhere(['or',
                ['like', 'full_name', $search],
                ['like', 'phone', $search],
                ['like', 'email', $search],
                ['like', 'game_name', $search]
            ]);
        }

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'sort' => [
                'defaultOrder' => ['created_at' => SORT_DESC],
                'attributes' => [
                    'created_at',
                    'status',
                    'full_name',
                    'date',
                ],
            ],
        ]);

        return $this->render('applications', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUpdateApplication($id)
    {
        $model = Application::findOne($id);

        if (!$model) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Статус заявки обновлен.');
            return $this->redirect(['applications']);
        }

        return $this->render('update-application', [
            'model' => $model,
        ]);
    }

    public function actionViewApplication($id)
    {
        $model = Application::findOne($id);

        if (!$model) {
            throw new NotFoundHttpException('Заявка не найдена.');
        }

        return $this->renderPartial('_view_application_admin', [
            'model' => $model,
        ]);
    }

    public function actionApproveApplication($id)
    {
        $model = Application::findOne($id);

        if ($model) {
            $model->status = Application::STATUS_APPROVED;
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Заявка одобрена.');
            }
        }

        return $this->redirect(['applications']);
    }

    public function actionUsers()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => User::find(),
            'sort' => ['defaultOrder' => ['created_at' => SORT_DESC]],
        ]);

        return $this->render('users', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionSchedule()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Schedule::find()->joinWith('game'),
            'sort' => ['defaultOrder' => ['date' => SORT_ASC, 'start_time' => SORT_ASC]],
        ]);

        return $this->render('schedule', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionUpdateSchedule($id)
    {
        $model = Schedule::findOne($id);

        if (!$model) {
            throw new NotFoundHttpException('Расписание не найдено.');
        }

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Расписание успешно обновлено.');
            return $this->redirect(['schedule']);
        }

        return $this->render('update-schedule', [
            'model' => $model,
        ]);
    }

    public function actionCreateSchedule()
    {
        $model = new Schedule();

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            Yii::$app->session->setFlash('success', 'Расписание успешно создано.');
            return $this->redirect(['schedule']);
        }

        return $this->render('create-schedule', [
            'model' => $model,
        ]);
    }

    public function actionDeleteSchedule($id)
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $model = Schedule::findOne($id);
        if ($model) {
            // Удаляем все заявки на это расписание
            Application::deleteAll(['schedule_id' => $id]);

            if ($model->delete()) {
                return ['success' => true];
            }
        }

        return ['success' => false, 'message' => 'Не удалось удалить расписание'];
    }

    public function actionDuplicateSchedule($id)
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $original = Schedule::findOne($id);
        if ($original) {
            $duplicate = new Schedule();
            $duplicate->attributes = $original->attributes;
            $duplicate->id = null;
            $duplicate->created_at = null;
            $duplicate->current_players = 0;

            if ($duplicate->save()) {
                return ['success' => true, 'id' => $duplicate->id];
            }
        }

        return ['success' => false, 'message' => 'Не удалось создать копию'];
    }

    public function actionReviews()
    {
        $dataProvider = new ActiveDataProvider([
            'query' => Review::find()->joinWith('user'),
            'sort' => ['defaultOrder' => ['created_at' => SORT_DESC]],
        ]);

        return $this->render('reviews', [
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionViewReview($id)
    {
        $model = Review::findOne($id);

        if (!$model) {
            throw new NotFoundHttpException('Отзыв не найден.');
        }

        return $this->renderPartial('_view_review', [
            'model' => $model,
        ]);
    }

    public function actionDeleteReview($id)
    {
        Yii::$app->response->format = \yii\web\Response::FORMAT_JSON;

        $model = Review::findOne($id);
        if ($model && $model->delete()) {
            return ['success' => true];
        }

        return ['success' => false, 'message' => 'Не удалось удалить отзыв'];
    }

    public function actionDeleteApplication($id)
    {
        $model = Application::findOne($id);

        if ($model && $model->delete()) {
            Yii::$app->session->setFlash('success', 'Заявка удалена.');
        } else {
            Yii::$app->session->setFlash('error', 'Не удалось удалить заявку.');
        }

        return $this->redirect(['applications']);
    }
}